<?php

namespace App\Http\Controllers\Admin\SK;

use App\Http\Controllers\Controller;
use App\Models\Penduduk;
use App\Models\Perangkat;
use App\Models\Profil;
use App\Models\RefSK;
use App\Models\SkKematian;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use RealRashid\SweetAlert\Facades\Alert;
use Yajra\DataTables\Facades\DataTables;

class KematianController extends Controller
{
    public function index(){
        $level = Auth::user()->level;
        $profil = Profil::where('pt_id', 1)->first();
        $refsks = RefSK::all();
        $penduduks = Penduduk::all();

        return view('admin.SK.kematian.index',[
            'level' => $level,
            'profil' => $profil,
            'refsks' => $refsks,
            'penduduks' => $penduduks,
        ])->with('tittle', 'Surat Keterangan Kematian' );
    }

    public function getData(Request $request)
    {
        if ($request->ajax()) {
            $kematian = SkKematian::query()
                        ->leftjoin('penduduks', 'penduduks.id', '=', 'sk_kematians.nik')
                        ->select(
                            'sk_kematians.id as kematian_id',
                            'sk_kematians.no_surat',
                            'sk_kematians.tgl_surat',
                            'sk_kematians.tgl_kematian',
                            'sk_kematians.jam',
                            'sk_kematians.sebab',
                            'sk_kematians.tempat_kematian',
                            'sk_kematians.tempat_pemakaman',
                            'sk_kematians.sumber',
                            'sk_kematians.hubungan',
                            'sk_kematians.nik as nik_meninggal',
                            'penduduks.nama_lengkap as nama_meninggal',
                        )          
                        ->orderBy('sk_kematians.id', 'desc');
            
            // Menghasilkan DataTables response
            return DataTables::of($kematian)
                ->addIndexColumn()

                // Menambahkan kolom opsi edit dan delete
                ->addColumn('opsi', function ($row) {
                    return '
                        <a href="'.route(Auth::user()->level . '-sk-kematian-cetak', $row->kematian_id).'" target="_blank" class="btn btn-icon btn-sm btn-outline btn-outline-primary text-hover-white mb-1 cetak">
                            <i class="bi bi-printer text-primary fs-4" data-bs-toggle="tooltip" title="Cetak"></i>
                        </a>
                        <a href="javascript:void(0)" data-id="'.$row->kematian_id.'" class="btn btn-icon btn-sm btn-outline btn-outline-info text-hover-white mb-1 edit-kematian">
                            <i class="bi bi-pencil text-info fs-4" data-bs-toggle="tooltip" data-bs-custom-class="tooltip-dark" data-bs-placement="top" title="Update"></i>
                        </a>
                        <a href="javascript:void(0)" data-id="'.$row->kematian_id.'" class="btn btn-icon btn-sm btn-outline btn-outline-danger text-hover-white mb-1 delete-kematian">
                            <i class="bi bi-trash text-danger fs-4" data-bs-toggle="tooltip" title="Hapus"></i>
                        </a>
                    ';
                })

                ->addColumn('nik_nama', function($row) {
                    return $row->nik_meninggal . '<br><strong>' . $row->nama_meninggal . '</strong>';
                })
                ->addColumn('tgl_surat_indo', function($row) {
                    return $row->no_surat . '<br>(' . date('d-m-Y', strtotime($row->tgl_surat)) . ')';
                })
                ->addColumn('tgl_kematian_indo', function($row) {
                    return date('d-m-Y', strtotime($row->tgl_kematian)).'<br>'.$row->jam.' WIB';
                })
                ->addColumn('sebab_tempat', function($row) {
                    return $row->sebab . '<br>di ' . $row->tempat_kematian;
                })
                ->addColumn('sumber', function($row) {
                    return $row->sumber . '<br>(' . $row->hubungan . ')';
                })
                // Menentukan kolom yang boleh mengandung HTML
                ->rawColumns(['opsi', 'nik_nama','sebab_tempat','sumber','tgl_surat_indo','tgl_kematian_indo'])

                ->make(true);
        }
    }

    public function store(Request $request)
    {
        // return $request;
        $request->validate([
            'nik'       => 'required|exists:penduduks,id',
            'no_surat'  => 'required|string',
            'tgl_surat' => 'required',
            'tgl_kematian' => 'required',
            'jam'   => 'required',
            'sebab'   => 'required',
            'tempat_kematian'   => 'required',
            'tempat_pemakaman'   => 'required',
            'sumber'   => 'required',
            'hubungan'   => 'required',
        ]);

        // Update status penduduk
        $penduduk = Penduduk::where('id',$request->nik)->first();
        $penduduk->update([
            'status' => "Meninggal",
        ]);

        $input = $request->all();
        $simpan = SkKematian::create($input);
        if ($simpan) {
            Alert::success('Berhasil', 'Data SK Kematian berhasil ditambahkan.');
        }else {
            Alert::error('Gagal', 'Data SK Kematian gagal ditambahkan.');
        }

        return redirect()->route('admin-sk-kematian-index');
    }

    public function getDataId($id)
    {
        $kematian = SkKematian::findOrFail($id);
        return response()->json($kematian);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'nik'       => 'required|exists:penduduks,id',
            'no_surat'  => 'required|string',
            'tgl_surat' => 'required',
            'tgl_kematian' => 'required',
            'jam'   => 'required',
            'sebab'   => 'required',
            'tempat_kematian'   => 'required',
            'tempat_pemakaman'   => 'required',
            'sumber'   => 'required',
            'hubungan'   => 'required',
        ]);
        $kematian = SkKematian::findOrFail($id);
        $simpan = $kematian->update([
            'nik'            => $request->nik,
            'no_surat'       => $request->no_surat,
            'tgl_surat'      => $request->tgl_surat,
            'tgl_kematian'      => $request->tgl_kematian,
            'jam'        => $request->jam,
            'sebab'        => $request->sebab,
            'tempat_kematian'        => $request->tempat_kematian,
            'tempat_pemakaman'        => $request->tempat_pemakaman,
            'sumber'        => $request->sumber,
            'hubungan'        => $request->hubungan,
        ]);
        if ($simpan) {
            Alert::success('Berhasil', 'Data SK Kematian berhasil diperbarui.');
        }else {
            Alert::error('Gagal', 'Data SK Kematian gagal diperbarui.');
        }

        return redirect()->route('admin-sk-kematian-index');
    }

    public function delete($id)
    {
        $kematian = SkKematian::findOrFail($id);

        // Update status penduduk
        $penduduk = Penduduk::where('id',$kematian->nik)->first();
        $penduduk->update([
            'status' => "Aktif",
        ]);

        $simpan = $kematian->delete();

        if ($simpan) {
            Alert::success('Berhasil', 'Data SK Kematian berhasil dihapus!');
        }else {
            Alert::error('Gagal', 'Data SK Kematian gagal dihapus!');
        }

        return redirect()->route('admin-sk-kematian-index');
    }

    public function cetak($id){
        $profil = Profil::where('pt_id', 1)->first();
        $data = SkKematian::where('id', $id)->first();
        $penduduk = Penduduk::where('id', $data->nik)->first();
        $kades = Perangkat::where('jabatan','Kepala Desa')->first();
        return view('admin.SK.kematian.template',[
            'profil' => $profil, 
            'data' => $data, 
            'penduduk' => $penduduk,
            'kades' => $kades,
        ]);
    }
}
